<?php
/**
 * Menu Manager Class
 *
 * Handles all operations related to WordPress menus.
 *
 * @package    WVC_Theme
 * @subpackage Core
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Class WVC_Menu_Manager
 *
 * Provides methods for creating, editing, deleting, and retrieving WordPress menus and menu items.
 */
class WVC_Menu_Manager {
    
    /**
     * Class instance
     *
     * @var WVC_Menu_Manager
     */
    private static $instance = null;
    
    /**
     * Get the singleton instance
     *
     * @return WVC_Menu_Manager
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Private constructor to enforce singleton pattern
    }
    
    /**
     * Get all menus
     *
     * @return array Array of menus
     */
    public function get_menus() {
        $menus = wp_get_nav_menus();
        $data = array();
        
        foreach ($menus as $menu) {
            $data[] = $this->prepare_menu_response($menu);
        }
        
        return $data;
    }
    
    /**
     * Get a single menu with its items
     *
     * @param int $menu_id Menu ID
     * @return array|WP_Error Menu data on success, WP_Error on failure
     */
    public function get_menu($menu_id) {
        // Check if menu exists
        $menu = wp_get_nav_menu_object($menu_id);
        if (!$menu) {
            return new WP_Error(
                'menu_not_found',
                __('Menu not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Get menu items
        $menu_items = wp_get_nav_menu_items($menu_id);
        
        // Prepare response
        return $this->prepare_menu_response($menu, $menu_items);
    }
    
    /**
     * Create a new menu
     *
     * @param array $args Menu data
     * @return array|WP_Error Menu data on success, WP_Error on failure
     */
    public function create_menu($args) {
        $name = $args['name'] ?? '';
        $location = $args['location'] ?? '';
        $items = $args['items'] ?? array();
        
        // Validate required fields
        if (empty($name)) {
            return new WP_Error(
                'missing_name',
                __('Menu name is required', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Check if menu with this name already exists
        $existing_menus = wp_get_nav_menus();
        $menu_exists = false;
        $menu_id = null;
        
        foreach ($existing_menus as $menu) {
            if ($menu->name === $name) {
                $menu_exists = true;
                $menu_id = $menu->term_id;
                break;
            }
        }
        
        // Create the menu if it doesn't exist
        if (!$menu_exists) {
            $menu_id = wp_create_nav_menu($name);
            
            if (is_wp_error($menu_id)) {
                return $menu_id; // Return WP_Error
            }
        }
        
        // Assign menu to location if provided
        if (!empty($location)) {
            $locations = get_theme_mod('nav_menu_locations');
            $locations[$location] = $menu_id;
            set_theme_mod('nav_menu_locations', $locations);
        }
        
        // Add menu items if provided
        $item_ids = array();
        if (!empty($items) && is_array($items)) {
            foreach ($items as $item) {
                $item_id = $this->add_menu_item($menu_id, $item);
                if (!is_wp_error($item_id)) {
                    $item_ids[] = $item_id;
                }
            }
        }
        
        return array(
            'success' => true,
            'id'      => $menu_id,
            'name'    => $name,
            'items'   => $item_ids,
            'existed' => $menu_exists,
        );
    }
    
    /**
     * Delete a menu
     *
     * @param int $menu_id Menu ID
     * @return array|WP_Error Delete status on success, WP_Error on failure
     */
    public function delete_menu($menu_id) {
        // Check if menu exists
        $menu = wp_get_nav_menu_object($menu_id);
        if (!$menu) {
            return new WP_Error(
                'menu_not_found',
                __('Menu not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Delete the menu
        $result = wp_delete_nav_menu($menu_id);
        
        if (!$result) {
            return new WP_Error(
                'menu_deletion_failed',
                __('Failed to delete menu', 'wvc-theme'),
                array('status' => 500)
            );
        }
        
        return array(
            'success' => true,
            'message' => sprintf(__('Menu "%s" deleted successfully', 'wvc-theme'), $menu->name),
        );
    }
    
    /**
     * Add a menu item to a menu
     *
     * @param int   $menu_id Menu ID
     * @param array $item    Item data
     * @return int|WP_Error  Menu item ID on success, WP_Error on failure
     */
    public function add_menu_item($menu_id, $item) {
        // Check if menu exists
        $menu = wp_get_nav_menu_object($menu_id);
        if (!$menu) {
            return new WP_Error(
                'menu_not_found',
                __('Menu not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Validate required fields
        if (empty($item['title'])) {
            return new WP_Error(
                'missing_title',
                __('Menu item title is required', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Default item args
        $args = array(
            'menu-item-title'  => $item['title'],
            'menu-item-status' => 'publish',
        );
        
        // Set menu item type and object
        if (!empty($item['object_id']) && !empty($item['object_type'])) {
            if ($item['object_type'] === 'post_type') {
                $post_type = get_post_type($item['object_id']);
                if (!$post_type) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid post ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $args['menu-item-type']      = 'post_type';
                $args['menu-item-object']    = $post_type;
                $args['menu-item-object-id'] = $item['object_id'];
            } elseif ($item['object_type'] === 'taxonomy') {
                $term = get_term($item['object_id']);
                if (is_wp_error($term)) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid taxonomy term ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $args['menu-item-type']      = 'taxonomy';
                $args['menu-item-object']    = $term->taxonomy;
                $args['menu-item-object-id'] = $item['object_id'];
            }
        } elseif (!empty($item['url'])) {
            $args['menu-item-type']   = 'custom';
            $args['menu-item-url']    = esc_url_raw($item['url']);
        } else {
            return new WP_Error(
                'missing_link_data',
                __('Menu item requires either an object_id and object_type, or a URL', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Set parent if provided
        if (!empty($item['parent'])) {
            $args['menu-item-parent-id'] = $item['parent'];
        }
        
        // Set order if provided
        if (isset($item['order'])) {
            $args['menu-item-position'] = $item['order'];
        }
        
        // Set target if provided
        if (isset($item['target'])) {
            $args['menu-item-target'] = $item['target'];
        }
        
        // Set classes if provided
        if (isset($item['classes'])) {
            $args['menu-item-classes'] = $item['classes'];
        }
        
        // Set description if provided
        if (isset($item['description'])) {
            $args['menu-item-description'] = $item['description'];
        }
        
        // Add the menu item
        $item_id = wp_update_nav_menu_item($menu_id, 0, $args);
        
        if (is_wp_error($item_id)) {
            return $item_id; // Return WP_Error
        }

        return $item_id;
    }
    
    /**
     * Edit a menu item
     *
     * @param int   $menu_id Menu ID
     * @param int   $item_id Menu item ID
     * @param array $args    Item data to update
     * @return array|WP_Error Updated item data on success, WP_Error on failure
     */
    public function edit_menu_item($menu_id, $item_id, $args) {
        // Check if menu exists
        $menu = wp_get_nav_menu_object($menu_id);
        if (!$menu) {
            return new WP_Error(
                'menu_not_found',
                __('Menu not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Check if menu item exists and belongs to the specified menu
        $menu_items = wp_get_nav_menu_items($menu_id);
        $item_exists = false;
        
        foreach ($menu_items as $menu_item) {
            if ($menu_item->ID == $item_id) {
                $item_exists = true;
                break;
            }
        }
        
        if (!$item_exists) {
            return new WP_Error(
                'menu_item_not_found',
                __('Menu item not found in the specified menu', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Get current menu item data
        $menu_item_type = get_post_meta($item_id, '_menu_item_type', true);
        $menu_item_object = get_post_meta($item_id, '_menu_item_object', true);
        $menu_item_object_id = get_post_meta($item_id, '_menu_item_object_id', true);
        
        // Prepare menu item data for update
        $update_args = array();
        
        // Basic fields
        if (isset($args['title'])) {
            $update_args['menu-item-title'] = $args['title'];
        }
        
        if (isset($args['description'])) {
            $update_args['menu-item-description'] = $args['description'];
        }
        
        if (isset($args['parent'])) {
            $update_args['menu-item-parent-id'] = $args['parent'];
        }
        
        if (isset($args['order'])) {
            $update_args['menu-item-position'] = $args['order'];
        }
        
        if (isset($args['target'])) {
            $update_args['menu-item-target'] = $args['target'];
        }
        
        if (isset($args['classes'])) {
            $update_args['menu-item-classes'] = $args['classes'];
        }
        
        // Handle object type and URL changes
        $object_type = isset($args['object_type']) ? $args['object_type'] : '';
        $object_id = isset($args['object_id']) ? $args['object_id'] : '';
        $url = isset($args['url']) ? $args['url'] : '';
        
        // If object type is being changed
        if (!empty($object_type) && $object_type !== $menu_item_type) {
            // Update menu item type
            $update_args['menu-item-type'] = $object_type;
            
            // Handle different object types
            if ($object_type === 'custom' && !empty($url)) {
                $update_args['menu-item-url'] = esc_url_raw($url);
                $update_args['menu-item-object'] = 'custom';
                $update_args['menu-item-object-id'] = 0;
            } elseif ($object_type === 'post_type' && !empty($object_id)) {
                $post_type = get_post_type($object_id);
                if (!$post_type) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid post ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $update_args['menu-item-object'] = $post_type;
                $update_args['menu-item-object-id'] = $object_id;
                $update_args['menu-item-url'] = '';
            } elseif ($object_type === 'taxonomy' && !empty($object_id)) {
                $term = get_term($object_id);
                if (is_wp_error($term)) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid taxonomy term ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $update_args['menu-item-object'] = $term->taxonomy;
                $update_args['menu-item-object-id'] = $object_id;
                $update_args['menu-item-url'] = '';
            }
        } 
        // If just updating object ID within the same type
        elseif (!empty($object_id) && $object_id != $menu_item_object_id) {
            if ($menu_item_type === 'post_type') {
                $post_type = get_post_type($object_id);
                if (!$post_type) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid post ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $update_args['menu-item-object'] = $post_type;
                $update_args['menu-item-object-id'] = $object_id;
            } elseif ($menu_item_type === 'taxonomy') {
                $term = get_term($object_id);
                if (is_wp_error($term)) {
                    return new WP_Error(
                        'invalid_object_id',
                        __('Invalid taxonomy term ID provided', 'wvc-theme'),
                        array('status' => 400)
                    );
                }
                $update_args['menu-item-object'] = $term->taxonomy;
                $update_args['menu-item-object-id'] = $object_id;
            }
        }
        // If only updating URL for custom menu item
        elseif ($menu_item_type === 'custom' && !empty($url)) {
            $update_args['menu-item-url'] = esc_url_raw($url);
        }
        
        // Check if there are any fields to update
        if (empty($update_args)) {
            return new WP_Error(
                'no_fields_to_update',
                __('No fields to update were provided', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Update the menu item
        $result = wp_update_nav_menu_item($menu_id, $item_id, $update_args);
        
        if (is_wp_error($result)) {
            return $result; // Return WP_Error
        }
        
        // Get the updated menu item
        $updated_item = get_post($item_id);
        $response_data = $this->prepare_menu_item_response($updated_item);
        
        return array(
            'success' => true,
            'menu_id' => $menu_id,
            'item'    => $response_data,
        );
    }
    
    /**
     * Delete a menu item
     *
     * @param int $menu_id Menu ID
     * @param int $item_id Menu item ID
     * @return array|WP_Error Delete status on success, WP_Error on failure
     */
    public function delete_menu_item($menu_id, $item_id) {
        // Check if menu exists
        $menu = wp_get_nav_menu_object($menu_id);
        if (!$menu) {
            return new WP_Error(
                'menu_not_found',
                __('Menu not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Check if menu item exists and belongs to the specified menu
        $menu_items = wp_get_nav_menu_items($menu_id);
        $item_exists = false;
        $item_title = '';
        
        foreach ($menu_items as $menu_item) {
            if ($menu_item->ID == $item_id) {
                $item_exists = true;
                $item_title = $menu_item->title;
                break;
            }
        }
        
        if (!$item_exists) {
            return new WP_Error(
                'menu_item_not_found',
                __('Menu item not found in the specified menu', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Delete the menu item
        $result = wp_delete_post($item_id, true);
        
        if (!$result) {
            return new WP_Error(
                'menu_item_deletion_failed',
                __('Failed to delete menu item', 'wvc-theme'),
                array('status' => 500)
            );
        }
        
        return array(
            'success' => true,
            'menu_id' => $menu_id,
            'item_id' => $item_id,
            'message' => sprintf(__('Menu item "%s" deleted successfully', 'wvc-theme'), $item_title),
        );
    }
    
    /**
     * Prepare menu response
     *
     * @param WP_Term $menu       The menu object
     * @param array   $menu_items Optional array of menu items
     * @return array Formatted menu data
     */
    public function prepare_menu_response($menu, $menu_items = null) {
        $data = array(
            'id'          => $menu->term_id,
            'name'        => $menu->name,
            'slug'        => $menu->slug,
            'description' => $menu->description,
            'count'       => $menu->count,
        );
        
        // Get menu locations
        $locations = get_nav_menu_locations();
        $menu_locations = array();
        
        foreach ($locations as $location => $menu_id) {
            if ($menu_id === $menu->term_id) {
                $menu_locations[] = $location;
            }
        }
        
        $data['locations'] = $menu_locations;
        
        // Add menu items if provided
        if ($menu_items) {
            $items = array();
            
            // Build hierarchical menu items array
            $menu_items_by_parent = array();
            foreach ($menu_items as $menu_item) {
                $parent = $menu_item->menu_item_parent ? $menu_item->menu_item_parent : 0;
                if (!isset($menu_items_by_parent[$parent])) {
                    $menu_items_by_parent[$parent] = array();
                }
                $menu_items_by_parent[$parent][] = $this->prepare_menu_item_response($menu_item);
            }
            
            // Get top level items
            if (isset($menu_items_by_parent[0])) {
                $items = $this->build_hierarchical_menu_items($menu_items_by_parent, 0);
            }
            
            $data['items'] = $items;
        }
        
        return $data;
    }
    
    /**
     * Build hierarchical menu items
     *
     * @param array $menu_items_by_parent Menu items grouped by parent
     * @param int   $parent_id           Parent ID
     * @return array Hierarchical menu items
     */
    private function build_hierarchical_menu_items($menu_items_by_parent, $parent_id) {
        $items = array();
        
        if (isset($menu_items_by_parent[$parent_id])) {
            foreach ($menu_items_by_parent[$parent_id] as $item) {
                if (isset($menu_items_by_parent[$item['id']])) {
                    $item['children'] = $this->build_hierarchical_menu_items($menu_items_by_parent, $item['id']);
                }
                $items[] = $item;
            }
        }
        
        return $items;
    }
    
    /**
     * Prepare menu item response
     *
     * @param WP_Post $menu_item The menu item object
     * @return array Formatted menu item data
     */
    public function prepare_menu_item_response($menu_item) {
        // Get menu item meta
        $menu_item_type = get_post_meta($menu_item->ID, '_menu_item_type', true);
        $object_id = get_post_meta($menu_item->ID, '_menu_item_object_id', true);
        $object = get_post_meta($menu_item->ID, '_menu_item_object', true);
        $url = '';
        
        // Get URL based on item type
        if ($menu_item_type === 'custom') {
            $url = get_post_meta($menu_item->ID, '_menu_item_url', true);
        } elseif ($menu_item_type === 'post_type') {
            $url = get_permalink($object_id);
        } elseif ($menu_item_type === 'taxonomy') {
            $url = get_term_link((int) $object_id, $object);
            if (is_wp_error($url)) {
                $url = '';
            }
        }
        
        return array(
            'id'           => $menu_item->ID,
            'title'        => $menu_item->title,
            'url'          => $url,
            'type'         => $menu_item_type,
            'object_id'    => (int) $object_id,
            'object'       => $object,
            'parent'       => (int) $menu_item->menu_item_parent,
            'order'        => (int) $menu_item->menu_order,
            'target'       => get_post_meta($menu_item->ID, '_menu_item_target', true),
            'classes'      => implode(' ', (array) get_post_meta($menu_item->ID, '_menu_item_classes', true)),
            'xfn'          => get_post_meta($menu_item->ID, '_menu_item_xfn', true),
            'description'  => $menu_item->description,
            'attr_title'   => $menu_item->attr_title,
        );
    }
}