<?php
/**
 * Page Manager Class
 *
 * Handles all operations related to WordPress pages.
 *
 * @package    WVC_Theme
 * @subpackage Core
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Class WVC_Page_Manager
 *
 * Provides methods for creating, editing, deleting, and retrieving WordPress pages.
 */
class WVC_Page_Manager {
    
    /**
     * Class instance
     *
     * @var WVC_Page_Manager
     */
    private static $instance = null;
    
    /**
     * Get the singleton instance
     *
     * @return WVC_Page_Manager
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Private constructor to enforce singleton pattern
    }
    
    /**
     * Create a new page
     *
     * @param array $args Page data
     * @return array|WP_Error Page data on success, WP_Error on failure
     */
    public function create_page($args) {
        $title = $args['title'] ?? '';
        $content = $args['content-managers'] ?? '';
        $template = $args['template'] ?? '';
        $status = $args['status'] ?? 'publish';
        $meta = $args['meta'] ?? array();
        $featured_image = $args['featured_image'] ?? null;
        $parent = $args['parent'] ?? 0;
        $menu_order = $args['menu_order'] ?? 0;
        
        // Validate required fields
        if (empty($title)) {
            return new WP_Error(
                'missing_title',
                __('Page title is required', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Prepare page data
        $page_data = array(
            'post_title'    => $title,
            'post_type'     => 'page',
            'post_status'   => $status,
        );
        
        // Add optional fields if provided
        if (!empty($content)) {
            $page_data['post_content'] = $content;
        }
        
        if (!empty($parent)) {
            $page_data['post_parent'] = $parent;
        }
        
        if (isset($menu_order)) {
            $page_data['menu_order'] = $menu_order;
        }
        
        // Insert the page
        $page_id = wp_insert_post($page_data, true);
        
        if (is_wp_error($page_id)) {
            return $page_id; // Return WP_Error
        }
        
        // Set template if provided
        if (!empty($template)) {
            update_post_meta($page_id, '_wp_page_template', $template);
        }
        
        // Set featured image if provided
        if (!empty($featured_image)) {
            set_post_thumbnail($page_id, $featured_image);
        }
        
        // Add meta data if provided
        if (!empty($meta) && is_array($meta)) {
            foreach ($meta as $meta_key => $meta_value) {
                update_post_meta($page_id, $meta_key, $meta_value);
            }
        }
        
        // Get the created page
        return $this->get_page($page_id);
    }
    
    /**
     * Get a page by ID
     *
     * @param int $page_id Page ID
     * @return array|WP_Error Page data on success, WP_Error on failure
     */
    public function get_page($page_id) {
        // Check if page exists
        $page = get_post($page_id);
        if (!$page || $page->post_type !== 'page') {
            return new WP_Error(
                'page_not_found',
                __('Page not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Prepare response
        $response = array(
            'id'           => $page->ID,
            'title'        => $page->post_title,
            'content-managers'      => $page->post_content,
            'excerpt'      => $page->post_excerpt,
            'status'       => $page->post_status,
            'date'         => $page->post_date,
            'modified'     => $page->post_modified,
            'author'       => (int) $page->post_author,
            'parent'       => (int) $page->post_parent,
            'menu_order'   => (int) $page->menu_order,
            'link'         => get_permalink($page->ID),
            'template'     => get_page_template_slug($page->ID),
            'featured_image' => has_post_thumbnail($page->ID) ? get_post_thumbnail_id($page->ID) : null,
        );
        
        // Get meta data
        $meta_keys = get_post_custom_keys($page->ID);
        if ($meta_keys) {
            $response['meta'] = array();
            
            foreach ($meta_keys as $key) {
                // Skip internal WordPress meta
                if (substr($key, 0, 1) === '_' && $key !== '_wp_page_template') {
                    continue;
                }
                
                $response['meta'][$key] = get_post_meta($page->ID, $key, true);
            }
        }
        
        return $response;
    }
    
    /**
     * Edit a page
     *
     * @param int   $page_id Page ID
     * @param array $args    Page data to update
     * @return array|WP_Error Updated page data on success, WP_Error on failure
     */
    public function edit_page($page_id, $args) {
        // Check if page exists
        $page = get_post($page_id);
        if (!$page || $page->post_type !== 'page') {
            return new WP_Error(
                'page_not_found',
                __('Page not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Prepare page data for update
        $page_data = array(
            'ID' => $page_id,
        );
        
        // Add fields that need to be updated
        if (isset($args['title'])) {
            $page_data['post_title'] = $args['title'];
        }
        
        if (isset($args['content-managers'])) {
            $page_data['post_content'] = $args['content-managers'];
        }
        
        if (isset($args['status'])) {
            $page_data['post_status'] = $args['status'];
        }
        
        if (isset($args['parent'])) {
            $page_data['post_parent'] = $args['parent'];
        }
        
        if (isset($args['menu_order'])) {
            $page_data['menu_order'] = $args['menu_order'];
        }
        
        // Check if there are any fields to update
        if (count($page_data) <= 1) {
            return new WP_Error(
                'no_fields_to_update',
                __('No fields to update were provided', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Update the page
        remove_filter('content_save_pre', 'wp_filter_post_kses');
        remove_filter('content_filtered_save_pre', 'wp_filter_post_kses');
        $result = wp_update_post($page_data, true);
        kses_init(); // re-add default KSES filters
        
        if (is_wp_error($result)) {
            return $result; // Return WP_Error
        }
        
        // Update template if provided
        if (isset($args['template'])) {
            update_post_meta($page_id, '_wp_page_template', $args['template']);
        }
        
        // Update featured image if provided
        if (isset($args['featured_image'])) {
            if (empty($args['featured_image'])) {
                delete_post_thumbnail($page_id);
            } else {
                set_post_thumbnail($page_id, $args['featured_image']);
            }
        }
        
        // Update meta data if provided
        if (isset($args['meta']) && is_array($args['meta'])) {
            foreach ($args['meta'] as $meta_key => $meta_value) {
                update_post_meta($page_id, $meta_key, $meta_value);
            }
        }
        
        // Get the updated page
        return $this->get_page($page_id);
    }
    
    /**
     * Delete a page
     *
     * @param int  $page_id Page ID
     * @param bool $force   Whether to bypass trash and delete permanently
     * @return array|WP_Error Delete status on success, WP_Error on failure
     */
    public function delete_page($page_id, $force = false) {
        // Check if page exists
        $page = get_post($page_id);
        if (!$page || $page->post_type !== 'page') {
            return new WP_Error(
                'page_not_found',
                __('Page not found', 'wvc-theme'),
                array('status' => 404)
            );
        }
        
        // Store page data for response
        $page_title = $page->post_title;
        $page_status = $page->post_status;
        
        // Delete the page
        $result = wp_delete_post($page_id, $force);
        
        if (!$result) {
            return new WP_Error(
                'page_deletion_failed',
                __('Failed to delete page', 'wvc-theme'),
                array('status' => 500)
            );
        }
        
        return array(
            'deleted'  => true,
            'id'       => $page_id,
            'title'    => $page_title,
            'previous' => array(
                'id'     => $page_id,
                'title'  => $page_title,
                'status' => $page_status,
            ),
        );
    }
    
    /**
     * Get all pages
     *
     * @param array $args Query arguments
     * @return array Array of pages
     */
    public function get_pages($args = array()) {
        $default_args = array(
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'menu_order',
            'order'          => 'ASC',
        );
        
        $query_args = wp_parse_args($args, $default_args);
        $pages = get_posts($query_args);
        $result = array();
        
        foreach ($pages as $page) {
            $result[] = $this->get_page($page->ID);
        }
        
        return $result;
    }
}