<?php
/**
 * Custom Post Type: Removed Post
 * 
 * This file defines a custom post type for storing removed posts
 * with their associated metadata in a dictionary format.
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Register the 'removed_post' custom post type
 */
function wvc_register_removed_post_type() {
    $labels = array(
        "name" => "Removed Posts",
        "singular_name" => "Removed Post",
        "menu_name" => "Removed Posts",
        "add_new" => "Add New",
        "add_new_item" => "Add New Removed Post",
        "edit_item" => "Edit Removed Post",
        "new_item" => "New Removed Post",
        "view_item" => "View Removed Post",
        "search_items" => "Search Removed Posts",
        "not_found" => "No removed posts found",
        "not_found_in_trash" => "No removed posts found in trash",
    );

    $args = array(
        "labels" => $labels,
        "public" => false, // Not publicly queryable
        "publicly_queryable" => false, // Not publicly queryable
        "show_ui" => false, // Not visible in admin menu
        "show_in_menu" => false, // Not visible in admin menu
        "show_in_nav_menus" => false, // Not available in navigation menus
        "show_in_admin_bar" => false, // Not visible in admin bar
        "show_in_rest" => false, // Not available in REST API
        "query_var" => false, // Not queryable via URL
        "rewrite" => false, // No URL rewriting
        "capability_type" => "post",
        "has_archive" => false, // No archive page
        "hierarchical" => false,
        "menu_position" => null,
        "menu_icon" => null,
        "supports" => array(
            "title",
            "content",
            "author",
            "custom_fields"
        ),
        "can_export" => false,
        "delete_with_user" => false,
    );

    register_post_type("wvc_removed_post", $args);
}

/**
 * Register meta fields for removed_post
 */
function wvc_register_removed_post_meta() {
    register_post_meta("wvc_removed_post", "wvc_original_post_type", array(
        "type" => "string",
        "description" => "The original post type before removal",
        "single" => true,
        "show_in_rest" => false,
        "auth_callback" => function() {
            return current_user_can("manage_options");
        }
    ));
    
}

/**
 * Initialize the custom post type and meta fields
 */
function wvc_init_removed_post() {
    add_action("init", "wvc_register_removed_post_type");
    add_action("init", "wvc_register_removed_post_meta");
}

// Initialize the custom post type
wvc_init_removed_post();

/**
 * Delete a post by changing its type to wvc_removed_post
 * 
 * @param int $post_id The ID of the post to delete
 * @return bool|WP_Error True on success, WP_Error on failure
 */
function wvc_delete_post($post_id) {
    // Get the original post
    $post = get_post($post_id);
    
    if (!$post) {
        return new WP_Error("post_not_found", "Post not found");
    }
    
    // Prevent deleting already removed posts
    if ($post->post_type === "wvc_removed_post") {
        return new WP_Error("already_removed", "Post is already removed");
    }
    
    // Store original post type
    $original_post_type = $post->post_type;
    
    
    // Update the post to wvc_removed_post type
    $update_data = array(
        "ID" => $post_id,
        "post_type" => "wvc_removed_post"
    );
    
    $result = wp_update_post($update_data);
    
    if (is_wp_error($result)) {
        return $result;
    }
    
    // Store original post type and existing metas
    update_post_meta($post_id, "wvc_original_post_type", $original_post_type);
    
    return true;
}

/**
 * Restore a removed post to its original post type
 * 
 * @param int $post_id The ID of the removed post to restore
 * @return bool|WP_Error True on success, WP_Error on failure
 */
function wvc_restore_post($post_id) {
    // Get the removed post
    $post = get_post($post_id);
    
    if (!$post) {
        return new WP_Error("post_not_found", "Post not found");
    }
    
    // Check if it's actually a removed post
    if ($post->post_type !== "wvc_removed_post") {
        return new WP_Error("not_removed", "Post is not a removed post");
    }
    
    // Get original post type
    $original_post_type = get_post_meta($post_id, "wvc_original_post_type", true);
    
    if (!$original_post_type) {
        return new WP_Error("no_original_type", "Original post type not found");
    }
        
    // Update the post back to original type
    $update_data = array(
        "ID" => $post_id,
        "post_type" => $original_post_type
    );
    
    $result = wp_update_post($update_data);
    
    if (is_wp_error($result)) {
        return $result;
    }

    
    // Clean up our special meta fields
    delete_post_meta($post_id, "wvc_original_post_type");
    
    return true;
}


