<?php

namespace WVC\FormSubmissions;

/**
 * WVC Form Submissions Admin Interface
 *
 * Admin interface customizations for form submissions
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.16
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Add custom columns to the form data admin list
 */
function wvc_add_form_data_columns($columns) {
    $new_columns = array();
    $new_columns['cb'] = $columns['cb'];
    $new_columns['title'] = $columns['title'];
    $new_columns['form_id'] = 'Form ID';
    $new_columns['section_name'] = 'Section';
    $new_columns['submission_date'] = 'Submitted';
    $new_columns['form_fields'] = 'Form Fields';
    
    return $new_columns;
}

/**
 * Display custom column content
 */
function wvc_display_form_data_columns($column, $post_id) {
    switch ($column) {
        case 'form_id':
            echo esc_html(get_post_meta($post_id, '_wvc_form_id', true));
            break;
        case 'section_name':
            echo esc_html(get_post_meta($post_id, '_wvc_section_name', true));
            break;
        case 'submission_date':
            $timestamp = get_post_meta($post_id, '_wvc_timestamp', true);
            if ($timestamp) {
                echo esc_html(date('Y-m-d H:i:s', $timestamp / 1000));
            }
            break;
        case 'form_fields':
            $form_data_json = get_post_meta($post_id, '_wvc_form_data', true);
            $form_data = $form_data_json ? json_decode($form_data_json, true) : array();
            if (!empty($form_data)) {
                echo esc_html(implode(', ', array_keys($form_data)));
            }
            break;
    }
}

/**
 * Add meta box for form submission details
 */
function wvc_add_form_submission_meta_box() {
    add_meta_box(
        'wvc_form_submission_details',
        'Form Submission Details',
        __NAMESPACE__ . '\wvc_form_submission_meta_box_callback',
        'wvc_form_data',
        'normal',
        'high'
    );
}

/**
 * Meta box callback to display form submission data
 */
function wvc_form_submission_meta_box_callback($post) {
    // Get all form submission data
    $submission_data = wvc_get_form_submission_data($post->ID);
    
    echo '<div style="margin: 15px 0;">';
    
    // Display basic information
    echo '<h3>Submission Information</h3>';
    echo '<table class="form-table">';
    echo '<tr><td><strong>Form ID:</strong></td><td>' . esc_html($submission_data['form_id']) . '</td></tr>';
    echo '<tr><td><strong>Section:</strong></td><td>' . esc_html($submission_data['section_name']) . '</td></tr>';
    echo '<tr><td><strong>Submission Date:</strong></td><td>' . esc_html($submission_data['submission_date']) . '</td></tr>';
    echo '<tr><td><strong>Session ID:</strong></td><td>' . esc_html($submission_data['session_id']) . '</td></tr>';
    echo '<tr><td><strong>Page URL:</strong></td><td><a href="' . esc_url($submission_data['page_url']) . '" target="_blank">' . esc_html($submission_data['page_url']) . '</a></td></tr>';
    echo '<tr><td><strong>User Agent:</strong></td><td>' . esc_html($submission_data['user_agent']) . '</td></tr>';
    echo '<tr><td><strong>Submission Attempt:</strong></td><td>' . esc_html($submission_data['submission_attempt']) . '</td></tr>';
    echo '<tr><td><strong>Form Version:</strong></td><td>' . esc_html($submission_data['form_version']) . '</td></tr>';
    echo '</table>';
    
    // Display form data
    echo '<h3>Form Data</h3>';
    if (!empty($submission_data['individual_fields'])) {
        echo '<table class="form-table">';
        foreach ($submission_data['individual_fields'] as $field_name => $field_value) {
            echo '<tr>';
            echo '<td style="width: 200px;"><strong>' . esc_html(ucfirst(str_replace('_', ' ', $field_name))) . ':</strong></td>';
            echo '<td>';
            if (strlen($field_value) > 100) {
                echo '<textarea readonly style="width: 100%; height: 80px;">' . esc_textarea($field_value) . '</textarea>';
            } else {
                echo esc_html($field_value);
            }
            echo '</td>';
            echo '</tr>';
        }
        echo '</table>';
    } else {
        echo '<p>No form data available.</p>';
    }
    
    // Display additional metadata if available
    if (!empty($submission_data['submission_metadata'])) {
        echo '<h3>Additional Information</h3>';
        echo '<table class="form-table">';
        foreach ($submission_data['submission_metadata'] as $key => $value) {
            echo '<tr><td><strong>' . esc_html(ucfirst(str_replace('_', ' ', $key))) . ':</strong></td><td>' . esc_html($value) . '</td></tr>';
        }
        echo '</table>';
    }
    
    // Display validation errors if any
    if (!empty($submission_data['validation_errors']) && $submission_data['validation_errors'] !== '{}') {
        echo '<h3>Validation Errors</h3>';
        echo '<pre style="background: #f1f1f1; padding: 10px; border-radius: 3px;">' . esc_html($submission_data['validation_errors']) . '</pre>';
    }
    
    echo '</div>';
}

/**
 * Remove default custom fields meta box for form submissions
 */
function wvc_remove_default_custom_fields() {
    remove_meta_box('postcustom', 'wvc_form_data', 'normal');
}

/**
 * Add custom CSS for form submission edit page
 */
function wvc_admin_form_submission_styles() {
    global $post_type;
    if ($post_type === 'wvc_form_data') {
        echo '<style>
            #postdivrich { display: none; } /* Hide content editor */
            .post-type-wvc_form_data #title { background-color: #f9f9f9; }
            .post-type-wvc_form_data #title:focus { background-color: #fff; }
            .wvc-readonly { background-color: #f9f9f9 !important; }
        </style>';
    }
}

/**
 * Add enhanced admin styles and JavaScript for form submissions
 */
function wvc_admin_enhanced_styles_scripts() {
    $screen = get_current_screen();
    if ($screen->post_type !== 'wvc_form_data' && 
        $screen->id !== 'wvc_form_data_page_wvc_form_overview' && 
        $screen->id !== 'admin_page_wvc_form_submissions') {
        return;
    }
    
    ?>
    <style>
        /* Form overview page styles */
        .wvc-forms-table .button {
            margin-right: 5px;
        }
        .wvc-stats-summary {
            border-left: 4px solid #0073aa;
            background: linear-gradient(135deg, #f1f1f1 0%, #e8e8e8 100%);
        }
        .wvc-forms-table table th,
        .wvc-forms-table table td {
            padding: 12px;
        }
        .wvc-forms-table tbody tr:hover {
            background-color: #f8f9fa;
            transform: translateY(-1px);
            transition: all 0.2s ease;
        }
        
        /* Form-specific submissions page styles */
        .wvc-form-info {
            margin: 20px 0; 
            padding: 15px; 
            background: #f8f9fa; 
            border-radius: 5px; 
            border-left: 4px solid #0073aa;
        }
        
        .wvc-breadcrumb {
            margin: 15px 0;
            font-size: 14px;
        }
        
        .wvc-breadcrumb a {
            text-decoration: none;
            color: #0073aa;
        }
        
        .wvc-breadcrumb a:hover {
            text-decoration: underline;
        }
        
        .wvc-dynamic-table {
            margin-top: 20px;
        }
        
        .wvc-submissions-table {
            border: 1px solid #e1e1e1;
        }
        
        .wvc-submissions-table th {
            background: #f8f9fa;
            font-weight: 600;
            border-bottom: 2px solid #dee2e6;
        }
        
        .wvc-submissions-table td {
            border-bottom: 1px solid #dee2e6;
            vertical-align: middle;
        }
        
        .wvc-submissions-table tbody tr:hover {
            background-color: #f8f9fa;
        }
        
        .wvc-submissions-table .button-small {
            font-size: 11px;
            padding: 2px 8px;
            height: auto;
            line-height: 1.4;
        }
        
        .wvc-table-actions {
            margin-top: 20px; 
            text-align: right;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 3px;
        }
        
        /* Filter dropdown styles */
        #form_id_filter {
            min-width: 200px;
            margin-left: 10px;
        }
        
        /* Enhanced notice styles */
        .notice.notice-info p {
            font-size: 14px;
        }
        .notice.notice-info a {
            text-decoration: none;
            font-weight: 500;
        }
        .notice.notice-info a:hover {
            text-decoration: underline;
        }
        
        /* Form overview cards */
        .wvc-form-card {
            background: #fff;
            border: 1px solid #e1e1e1;
            border-radius: 8px;
            padding: 20px;
            margin: 10px 0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        .wvc-form-card:hover {
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
            transform: translateY(-2px);
        }
        .wvc-form-card h3 {
            margin-top: 0;
            color: #0073aa;
        }
        .wvc-form-card .stats {
            display: flex;
            justify-content: space-between;
            margin: 15px 0;
        }
        .wvc-form-card .stat-item {
            text-align: center;
        }
        .wvc-form-card .stat-number {
            font-size: 24px;
            font-weight: bold;
            color: #0073aa;
        }
        .wvc-form-card .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
        }
        
        /* Loading states */
        .wvc-loading {
            opacity: 0.6;
            pointer-events: none;
        }
        
        /* Mobile responsive */
        @media (max-width: 768px) {
            .wvc-forms-table table {
                font-size: 12px;
            }
            .wvc-forms-table table th,
            .wvc-forms-table table td {
                padding: 8px;
            }
            .wvc-form-card .stats {
                flex-direction: column;
            }
            .wvc-form-card .stat-item {
                margin: 5px 0;
            }
        }
    </style>
    
    <script>
        jQuery(document).ready(function($) {
            // Auto-submit form filter when changed
            $('#form_id_filter').on('change', function() {
                var formId = $(this).val();
                var currentUrl = window.location.href.split('?')[0];
                var newUrl = currentUrl + '?post_type=wvc_form_data';
                
                if (formId) {
                    newUrl += '&form_id=' + encodeURIComponent(formId);
                }
                
                window.location.href = newUrl;
            });
            
            // Add loading state to buttons
            $('.wvc-forms-table .button').on('click', function() {
                $(this).addClass('wvc-loading').text('Loading...');
            });
            
            // Enhance form cards with click-to-view functionality
            $('.wvc-form-card').on('click', function(e) {
                if (!$(e.target).is('a, button')) {
                    var viewUrl = $(this).find('.button-primary').attr('href');
                    if (viewUrl) {
                        window.location.href = viewUrl;
                    }
                }
            });
            
            // Add tooltips to statistics
            $('.stat-item').each(function() {
                var $this = $(this);
                var label = $this.find('.stat-label').text();
                var number = $this.find('.stat-number').text();
                $this.attr('title', label + ': ' + number);
            });
            
            // Highlight current filter in dropdown
            var urlParams = new URLSearchParams(window.location.search);
            var currentFormId = urlParams.get('form_id');
            if (currentFormId) {
                $('#form_id_filter').val(currentFormId);
                $('#form_id_filter').css('background-color', '#fff3cd');
            }
        });
    </script>
    <?php
}

/**
 * Add dashboard widget for recent form submissions
 */
function wvc_add_dashboard_widget() {
    wp_add_dashboard_widget(
        'wvc_recent_form_submissions',
        'Recent Form Submissions',
        __NAMESPACE__ . '\wvc_dashboard_widget_content'
    );
}

/**
 * Dashboard widget content
 */
function wvc_dashboard_widget_content() {
    $recent_submissions = get_posts(array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => 5,
        'orderby' => 'date',
        'order' => 'DESC'
    ));
    
    if (empty($recent_submissions)) {
        echo '<p>No form submissions yet.</p>';
        return;
    }
    
    echo '<table style="width: 100%;">';
    echo '<thead><tr><th>Form</th><th>Date</th><th>Fields</th><th>Action</th></tr></thead>';
    echo '<tbody>';
    
    foreach ($recent_submissions as $submission) {
        $form_id = get_post_meta($submission->ID, '_wvc_form_id', true);
        $timestamp = get_post_meta($submission->ID, '_wvc_timestamp', true);
        $form_data_json = get_post_meta($submission->ID, '_wvc_form_data', true);
        $form_data = $form_data_json ? json_decode($form_data_json, true) : array();
        
        $date = $timestamp ? date('M j, Y H:i', $timestamp / 1000) : get_the_date('M j, Y H:i', $submission->ID);
        $fields_preview = !empty($form_data) ? implode(', ', array_keys($form_data)) : 'No fields';
        if (strlen($fields_preview) > 40) {
            $fields_preview = substr($fields_preview, 0, 37) . '...';
        }
        
        echo '<tr>';
        echo '<td>' . esc_html($form_id) . '</td>';
        echo '<td>' . esc_html($date) . '</td>';
        echo '<td>' . esc_html($fields_preview) . '</td>';
        echo '<td><a href="' . admin_url('post.php?post=' . $submission->ID . '&action=edit') . '">View</a></td>';
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    
    echo '<p style="text-align: right; margin-top: 10px;">';
    echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data') . '">View All Form Submissions &raquo;</a>';
    echo '</p>';
}

/**
 * Add custom Form Data menu with only Form Overview
 */
function wvc_add_form_overview_menu() {
    // Add main menu page
    add_menu_page(
        'Form Overview',           // Page title
        'Form Data',              // Menu title
        'manage_options',         // Capability
        'wvc_form_overview',      // Menu slug
        __NAMESPACE__ . '\wvc_form_overview_page', // Callback function
        'dashicons-forms',        // Icon
        30                        // Position
    );
    
    // Add Form Submissions submenu (hidden from menu but accessible via URL)
    add_submenu_page(
        null, // Hide from menu
        'Form Submissions',
        'Form Submissions',
        'manage_options',
        'wvc_form_submissions',
        __NAMESPACE__ . '\wvc_form_submissions_page'
    );
}

/**
 * Remove "Add New" button from form submissions list
 */
function wvc_remove_add_new_button() {
    global $submenu;
    
    // Remove "Add New" submenu for wvc_form_data post type
    if (isset($submenu['edit.php?post_type=wvc_form_data'])) {
        foreach ($submenu['edit.php?post_type=wvc_form_data'] as $key => $menu_item) {
            if ($menu_item[2] === 'post-new.php?post_type=wvc_form_data') {
                unset($submenu['edit.php?post_type=wvc_form_data'][$key]);
                break;
            }
        }
    }
}

/**
 * Hide "Add New" button with CSS on form submissions page
 */
function wvc_hide_add_new_button_css() {
    global $pagenow, $typenow;
    
    if ($pagenow === 'edit.php' && $typenow === 'wvc_form_data') {
        echo '<style>
            .page-title-action, 
            .add-new-h2, 
            .wp-heading-inline + .page-title-action {
                display: none !important;
            }
        </style>';
    }
}

/**
 * Handle redirect to form submissions list
 */
function wvc_redirect_to_submissions() {
    $redirect_url = admin_url('edit.php?post_type=wvc_form_data');
    wp_redirect($redirect_url);
    exit;
}

/**
 * Display form overview page with distinct form IDs and their submission counts
 */
function wvc_form_overview_page() {
    // Get distinct form IDs with counts
    $form_stats = wvc_get_distinct_form_ids_with_counts();
    $total_submissions = wvc_get_total_submissions_count();

    echo '<div class="wrap">';
    echo '<h1>Form Overview</h1>';
    
    // Add warning banner about publishing requirement
    echo '<div class="notice notice-warning" style="margin: 15px 0; padding: 12px 15px; border-left: 4px solid #ffb900; background-color: #fff8e5;">';
    echo '<p style="margin: 0; font-size: 14px;"><strong>Note:</strong> To store form submissions, please publish your site first. After publishing, submissions will work from the live website.</p>';
    echo '</div>';
    
    if (empty($form_stats)) {
        echo '<div class="notice notice-info"><p>No form submissions found.</p></div>';
        echo '</div>';
        return;
    }
    
    
    
    // Fallback table for users who prefer table layout
    echo '<div style="margin-top: 30px;">';
    echo '<h3>Detailed Table View</h3>';
    echo '<div class="wvc-forms-table">';
    echo '<table class="wp-list-table widefat fixed striped">';
    echo '<thead>';
    echo '<tr>';
    echo '<th scope="col" style="width: 40%;">Form ID</th>';
    echo '<th scope="col" style="width: 30%;">Submissions</th>';
    echo '<th scope="col" style="width: 30%;">Actions</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    foreach ($form_stats as $form_stat) {
        $form_id = esc_html($form_stat['form_id']);
        $submission_count = $form_stat['submission_count'];
        $percentage = $total_submissions > 0 ? round(($submission_count / $total_submissions) * 100, 1) : 0;
        
        // Generate URLs for viewing submissions
        $view_submissions_url = admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_stat['form_id']));
        
        echo '<tr>';
        echo '<td><strong>' . $form_id . '</strong></td>';
        echo '<td>' . $submission_count . '</td>';
        echo '<td>';
        echo '<a href="' . esc_url($view_submissions_url) . '" class="button button-primary">View Submissions</a>';
        echo '</td>';
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
    echo '</div>';
    
    echo '</div>';
}

/**
 * Display form-specific submissions page with dynamic columns
 */
function wvc_form_submissions_page() {
    // Get form ID from URL parameter
    $form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';
    
    if (empty($form_id)) {
        echo '<div class="wrap">';
        echo '<h1>Form Submissions</h1>';
        echo '<div class="notice notice-error"><p>No form ID specified. Please select a form from the <a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">Form Overview</a> page.</p></div>';
        echo '</div>';
        return;
    }
    
    // Get form fields from the first submission
    $form_fields = wvc_get_form_fields_by_form_id($form_id);
    
    // Get all submissions for this form
    $submissions = wvc_get_form_submissions_with_fields($form_id);
    
    // Get submission count
    $submission_count = wvc_get_form_submission_count($form_id);
    
    echo '<div class="wrap">';
    echo '<h1>Form Submissions: ' . esc_html($form_id) . '</h1>';
    
    // Add navigation breadcrumb
    echo '<p class="wvc-breadcrumb">';
    echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">← Back to Form Overview</a>';
    // echo ' | <a href="' . admin_url('edit.php?post_type=wvc_form_data') . '">All Submissions</a>';
    echo '</p>';

    // Add warning banner about publishing requirement
    echo '<div class="notice notice-warning" style="margin: 15px 0; padding: 12px 15px; border-left: 4px solid #ffb900; background-color: #fff8e5;">';
    echo '<p style="margin: 0; font-size: 14px;"><strong>Note:</strong> To store form submissions, please publish your site first. After publishing, submissions will work from the live website.</p>';
    echo '</div>';
    
    // Display form information
    echo '<div class="wvc-form-info" style="margin: 20px 0; padding: 15px; background: #f8f9fa; border-radius: 5px; border-left: 4px solid #0073aa;">';
    echo '<h3>Form Information</h3>';
    echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">';
    echo '<div><strong>Form ID:</strong> ' . esc_html($form_id) . '</div>';
    echo '<div><strong>Total Submissions:</strong> ' . $submission_count . '</div>';
    echo '<div><strong>Form Fields:</strong> ' . count($form_fields) . '</div>';
    echo '</div>';
    echo '</div>';
    
    if (empty($submissions)) {
        echo '<div class="notice notice-info"><p>No submissions found for this form.</p></div>';
        echo '</div>';
        return;
    }
    
    // Display the dynamic table
    echo '<div class="wvc-dynamic-table">';
    echo '<table class="wp-list-table widefat fixed striped wvc-submissions-table">';
    echo '<thead>';
    echo '<tr>';
    echo '<th scope="col" style="width: 60px;">#</th>';
    echo '<th scope="col" style="width: 120px;">Submission Date</th>';
    
    // Add dynamic columns based on form fields
    foreach ($form_fields as $field) {
        $field_label = ucwords(str_replace('_', ' ', $field));
        echo '<th scope="col">' . esc_html($field_label) . '</th>';
    }
    
    echo '<th scope="col" style="width: 80px;">Actions</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $row_number = 1;
    foreach ($submissions as $submission) {
        echo '<tr>';
        echo '<td>' . $row_number . '</td>';
        echo '<td>' . esc_html($submission['submission_date']) . '</td>';
        
        // Add dynamic field data
        foreach ($form_fields as $field) {
            $field_value = isset($submission['individual_fields'][$field]) ? $submission['individual_fields'][$field] : '';
            
            // Truncate long values for table display
           /* if (strlen($field_value) > 50) {
                $display_value = substr($field_value, 0, 47) . '...';
            } else {
                $display_value = $field_value;
            }*/
            
            echo '<td title="' . esc_attr($field_value) . '">' . esc_html($field_value) . '</td>';
        }
        
        // Add actions column
        $submission_post = get_post($submission['form_id'] ? get_posts(array(
            'post_type' => 'wvc_form_data',
            'meta_query' => array(
                array('key' => '_wvc_form_id', 'value' => $submission['form_id']),
                array('key' => '_wvc_timestamp', 'value' => $submission['timestamp'])
            ),
            'posts_per_page' => 1
        ))[0]->ID ?? 0 : 0);
        
        // Find the actual post ID for this submission
        $actual_post_id = 0;
        if (!empty($submission['timestamp'])) {
            $posts = get_posts(array(
                'post_type' => 'wvc_form_data',
                'meta_query' => array(
                    array('key' => '_wvc_form_id', 'value' => $form_id),
                    array('key' => '_wvc_timestamp', 'value' => $submission['timestamp'])
                ),
                'posts_per_page' => 1
            ));
            if (!empty($posts)) {
                $actual_post_id = $posts[0]->ID;
            }
        }
        
        echo '<td>';
        
            echo '<button class="button button-small button-primary" disabled style="opacity: 0.5; cursor: not-allowed;">Edit</button>';
        
        echo '</td>';
        echo '</tr>';
        
        $row_number++;
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
    
    // Add export functionality
    echo '<div class="wvc-table-actions" style="margin-top: 20px; text-align: right;">';
    echo '<p><strong>Total Submissions:</strong> ' . count($submissions) . '</p>';
    echo '</div>';
    
    echo '</div>';
}

/**
 * Add form ID filter to the form submissions list
 */
function wvc_add_form_id_filter() {
    global $typenow;
    
    if ($typenow !== 'wvc_form_data') {
        return;
    }
    
    // Get distinct form IDs for the dropdown
    $form_stats = wvc_get_distinct_form_ids_with_counts();
    $selected_form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';
    
    echo '<select name="form_id" id="form_id_filter">';
    echo '<option value="">All Forms</option>';
    
    foreach ($form_stats as $form_stat) {
        $form_id = esc_attr($form_stat['form_id']);
        $selected = selected($selected_form_id, $form_id, false);
        echo '<option value="' . $form_id . '"' . $selected . '>' . esc_html($form_id) . ' (' . $form_stat['submission_count'] . ')</option>';
    }
    
    echo '</select>';
}

/**
 * Filter form submissions by form ID
 */
function wvc_filter_submissions_by_form_id($query) {
    global $pagenow, $typenow;
    
    if ($pagenow === 'edit.php' && $typenow === 'wvc_form_data' && isset($_GET['form_id']) && !empty($_GET['form_id'])) {
        $form_id = sanitize_text_field($_GET['form_id']);
        
        $query->set('meta_key', '_wvc_form_id');
        $query->set('meta_value', $form_id);
    }
}

/**
 * Add admin notice with form submission statistics on form data pages
 */
function wvc_admin_form_stats_notice() {
    $screen = get_current_screen();
    if ($screen->post_type !== 'wvc_form_data') {
        return;
    }
    
    // Check if we're filtering by form ID
    $current_form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';
    
    if ($current_form_id) {
        // Show statistics for the selected form
        $form_count = wvc_get_form_submission_count($current_form_id);
        echo '<div class="notice notice-info">';
        echo '<p><strong>Showing submissions for Form ID: ' . esc_html($current_form_id) . '</strong> ';
        echo '(' . $form_count . ' submissions) | ';
        echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data') . '">View All Forms</a> | ';
        echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">Form Overview</a>';
        echo '</p>';
        echo '</div>';
    } else {
        // Get submission statistics
        $total_submissions = wp_count_posts('wvc_form_data')->private;
        $today_submissions = get_posts(array(
            'post_type' => 'wvc_form_data',
            'post_status' => 'private',
            'date_query' => array(
                array(
                    'after' => 'today'
                )
            ),
            'fields' => 'ids',
            'posts_per_page' => -1
        ));
        $today_count = count($today_submissions);
        
        $this_week_submissions = get_posts(array(
            'post_type' => 'wvc_form_data',
            'post_status' => 'private',
            'date_query' => array(
                array(
                    'after' => '1 week ago'
                )
            ),
            'fields' => 'ids',
            'posts_per_page' => -1
        ));
        $week_count = count($this_week_submissions);
        
        echo '<div class="notice notice-info">';
        echo '<p><strong>Form Submission Statistics:</strong> ';
        echo 'Total: ' . $total_submissions . ' | ';
        echo 'Today: ' . $today_count . ' | ';
        echo 'This Week: ' . $week_count . ' | ';
        echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">View Form Overview</a>';
        echo '</p>';
        echo '</div>';
    }
}