<?php
/**
 * Delete Restore REST API Class
 *
 * Provides REST API endpoints for deleting and restoring WordPress posts and terms.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/delete_restore/post.php';

/**
 * Class WVC_Delete_Restore_REST_API
 *
 * Handles delete and restore operations via REST API
 */
class WVC_Delete_Restore_REST_API extends WVC_REST_API
{
    /**
     * Delete Restore Manager instance
     *
     */

    const DELETE_POSTS_CAP = "delete_posts";

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Delete endpoint
        register_rest_route(
            $this->namespace,
            '/wvc-delete',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'delete_item'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::DELETE_POSTS_CAP);
                },
                'args'                => array(
                    'type' => array(
                        'required'    => true,
                        'type'        => 'string',
                        'enum'        => array('post', 'term'),
                        'description' => __('Type of item to delete (post or term)', 'wvc-theme'),
                    ),
                    'id'   => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('ID of the item to delete', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Restore endpoint
        register_rest_route(
            $this->namespace,
            '/wvc-restore',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'restore_item'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::DELETE_POSTS_CAP);
                },
                'args'                => array(
                    'type' => array(
                        'required'    => true,
                        'type'        => 'string',
                        'enum'        => array('post', 'term'),
                        'description' => __('Type of item to restore (post or term)', 'wvc-theme'),
                    ),
                    'id'   => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('ID of the item to restore', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Test endpoint to verify REST API is working
        register_rest_route(
            $this->namespace,
            '/test',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => function () {
                    return new WP_REST_Response(
                        array(
                            'message'   => 'WVC REST API is working',
                            'timestamp' => current_time('mysql')
                        ), 
                        200
                    );
                },
                'permission_callback' => '__return_true'
            )
        );
    }

    /**
     * Handle delete item request
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_item($request)
    {
        $type = $request->get_param("type");
        $id = $request->get_param("id");

        // Fallback: try to get from JSON body if parameters are empty
        if (empty($type) || empty($id)) {
            $body = $request->get_body();
            $json_data = json_decode($body, true);

            if ($json_data) {
                $type = isset($json_data['type']) ? $json_data['type'] : $type;
                $id = isset($json_data['id']) ? $json_data['id'] : $id;
            }
        }

        if ($type === "term") {
            return new WP_REST_Response(array(
                "message" => "Term deletion not implemented yet"
            ), 501);
        }

        if ($type === "post") {
            $result = wvc_delete_post($id);

            if (is_wp_error($result)) {
                $error_code = $result->get_error_code();

                if ($error_code === "post_not_found") {
                    return new WP_REST_Response(array(
                        "message" => "Post not found"
                    ), 404);
                }

                return new WP_REST_Response(array(
                    "message" => $result->get_error_message()
                ), 500);
            }

            return new WP_REST_Response(array(
                "message" => "Post deleted successfully",
                "id" => $id
            ), 200);
        }

        return new WP_REST_Response(array(
            "message" => "Invalid type parameter"
        ), 400);

    }

    /**
     * Handle restore item request
     *
     * @param WP_REST_Request $request The request object
     * @return WP_REST_Response|WP_Error
     */
    public function restore_item($request)
    {
        $type = $request->get_param("type");
        $id = $request->get_param("id");

        // Fallback: try to get from JSON body if parameters are empty
        if (empty($type) || empty($id)) {
            $body = $request->get_body();
            $json_data = json_decode($body, true);

            if ($json_data) {
                $type = isset($json_data['type']) ? $json_data['type'] : $type;
                $id = isset($json_data['id']) ? $json_data['id'] : $id;
            }
        }

        if ($type === "term") {
            return new WP_REST_Response(array(
                "message" => "Term restoration not implemented yet"
            ), 501);
        }

        if ($type === "post") {
            $result = wvc_restore_post($id);

            if (is_wp_error($result)) {
                $error_code = $result->get_error_code();

                if ($error_code === "post_not_found") {
                    return new WP_REST_Response(array(
                        "message" => "Post not found"
                    ), 404);
                }

                return new WP_REST_Response(array(
                    "message" => $result->get_error_message()
                ), 500);
            }

            return new WP_REST_Response(array(
                "message" => "Post restored successfully",
                "id" => $id
            ), 200);
        }

        return new WP_REST_Response(array(
            "message" => "Invalid type parameter"
        ), 400);
    }
}

// Initialize the Delete Restore REST API
new WVC_Delete_Restore_REST_API();