<?php
/**
 * Custom Post Type Meta Box Handler
 *
 * Dynamically renders custom fields for custom post types based on field schemas
 * stored in wp_options. Handles rendering and saving of custom field data to
 * individual meta keys (wvc_{post_type}_{field_key}).
 *
 * @package    WVC_Theme
 * @subpackage Includes
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class WVC_CPT_Meta_Box
 *
 * Handles dynamic meta box registration and rendering for custom post types
 */
class WVC_CPT_Meta_Box
{
    /**
     * Instance of this class
     *
     * @var WVC_CPT_Meta_Box
     */
    private static $instance = null;

    /**
     * Get singleton instance
     *
     * @return WVC_CPT_Meta_Box
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action('add_meta_boxes', array($this, 'register_meta_boxes'));
        add_action('save_post', array($this, 'save_meta_box_data'), 10, 2);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Register meta boxes for all custom post types with field schemas
     */
    public function register_meta_boxes()
    {
        $cpt_configs = get_option('wvc_custom_post_types', array());

        foreach ($cpt_configs as $post_type_key => $config) {
            // Check if this CPT has custom fields
            $fields = get_option("wvc_cpt_fields_{$post_type_key}", array());

            if (!empty($fields)) {
                add_meta_box(
                    "wvc_cpt_fields_{$post_type_key}",
                    __('Custom Fields', 'wvc-theme'),
                    array($this, 'render_meta_box'),
                    $post_type_key,
                    'normal',
                    'high',
                    array('fields' => $fields)
                );
            }
        }
    }

    /**
     * Render the meta box with dynamic fields
     *
     * @param WP_Post $post    The post object.
     * @param array   $metabox The metabox array with field definitions.
     */
    public function render_meta_box($post, $metabox)
    {
        $fields = $metabox['args']['fields'] ?? array();

        if (empty($fields)) {
            echo '<p>' . __('No custom fields defined for this post type.', 'wvc-theme') . '</p>';
            return;
        }

        // Add nonce for security
        wp_nonce_field('wvc_cpt_meta_box_nonce', 'wvc_cpt_meta_box_nonce');

        echo '<div class="wvc-custom-fields-wrapper">';

        $post_type = $post->post_type;

        foreach ($fields as $field) {
            $field_key = $field['key'];
            $meta_key = "wvc_{$post_type}_{$field_key}";
            $field_value = get_post_meta($post->ID, $meta_key, true);

            // Use default value if meta doesn't exist
            if ($field_value === '') {
                $field_value = $field['default_value'] ?? '';
            }

            echo '<div class="wvc-field-row" style="margin-bottom: 20px;">';

            // Field label
            $required_mark = !empty($field['required']) ? ' <span class="required" style="color: red;">*</span>' : '';
            echo '<label for="wvc_field_' . esc_attr($field_key) . '" style="display: block; margin-bottom: 5px; font-weight: 600;">';
            echo esc_html($field['label']) . $required_mark;
            echo '</label>';

            // Render field based on type
            $this->render_field($field, $field_value, $post->ID);

            // Help text
            if (!empty($field['help_text'])) {
                echo '<p class="description" style="margin-top: 3px; margin-bottom: 0;">';
                echo esc_html($field['help_text']);
                echo '</p>';
            }

            echo '</div>';
        }

        echo '</div>';
    }

    /**
     * Render individual field based on type
     *
     * @param array  $field       The field definition.
     * @param mixed  $value       The current field value.
     * @param int    $post_id     The post ID.
     */
    private function render_field($field, $value, $post_id)
    {
        $field_key = $field['key'];
        $field_type = $field['type'];
        $field_name = "wvc_cpt_data[{$field_key}]";
        $field_id = "wvc_field_{$field_key}";
        $required = !empty($field['required']) ? 'required' : '';
        $placeholder = !empty($field['placeholder']) ? 'placeholder="' . esc_attr($field['placeholder']) . '"' : '';

        switch ($field_type) {
            case 'text':
                echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $required . ' class="regular-text" />';
                break;

            case 'textarea':
                echo '<textarea id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" rows="5" ' . $placeholder . ' ' . $required . ' class="large-text">' . esc_textarea($value) . '</textarea>';
                break;

            case 'url':
                echo '<input type="url" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_url($value) . '" ' . $placeholder . ' ' . $required . ' class="regular-text" />';
                break;

            case 'email':
                echo '<input type="email" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $required . ' class="regular-text" />';
                break;

            case 'date':
                echo '<input type="date" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" ' . $required . ' class="regular-text" />';
                break;

            case 'number':
                echo '<input type="number" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $required . ' class="regular-text" />';
                break;

            case 'select':
                echo '<select id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" ' . $required . ' class="regular-text">';
                echo '<option value="">' . __('Select an option...', 'wvc-theme') . '</option>';
                if (!empty($field['options']) && is_array($field['options'])) {
                    foreach ($field['options'] as $option) {
                        $selected = ($value === $option) ? 'selected' : '';
                        echo '<option value="' . esc_attr($option) . '" ' . $selected . '>' . esc_html($option) . '</option>';
                    }
                }
                echo '</select>';
                break;

            case 'checkbox':
                $checked = !empty($value) ? 'checked' : '';
                echo '<label for="' . esc_attr($field_id) . '">';
                echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="1" ' . $checked . ' />';
                echo ' ' . __('Yes', 'wvc-theme');
                echo '</label>';
                break;

            case 'image':
                $image_url = $value ? wp_get_attachment_url($value) : '';
                echo '<div class="wvc-image-field">';
                echo '<input type="hidden" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" />';
                echo '<div class="wvc-image-preview" style="margin-bottom: 10px;">';
                if ($image_url) {
                    echo '<img src="' . esc_url($image_url) . '" style="max-width: 300px; height: auto; display: block;" />';
                }
                echo '</div>';
                echo '<button type="button" class="button wvc-upload-image-button" data-field-id="' . esc_attr($field_id) . '">' . __('Upload Image', 'wvc-theme') . '</button> ';
                echo '<button type="button" class="button wvc-remove-image-button" data-field-id="' . esc_attr($field_id) . '" style="' . ($image_url ? '' : 'display:none;') . '">' . __('Remove Image', 'wvc-theme') . '</button>';
                echo '</div>';
                break;

            default:
                echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" value="' . esc_attr($value) . '" class="regular-text" />';
        }
    }

    /**
     * Save meta box data
     *
     * @param int     $post_id The post ID.
     * @param WP_Post $post    The post object.
     */
    public function save_meta_box_data($post_id, $post)
    {
        // Security checks
        if (!isset($_POST['wvc_cpt_meta_box_nonce']) || !wp_verify_nonce($_POST['wvc_cpt_meta_box_nonce'], 'wvc_cpt_meta_box_nonce')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Check if this post type has custom fields
        $post_type = $post->post_type;
        $fields = get_option("wvc_cpt_fields_{$post_type}", array());

        if (empty($fields)) {
            return;
        }

        // Save each field individually
        if (isset($_POST['wvc_cpt_data']) && is_array($_POST['wvc_cpt_data'])) {
            foreach ($fields as $field) {
                $field_key = $field['key'];
                $field_type = $field['type'];
                $meta_key = "wvc_{$post_type}_{$field_key}";

                // Check if required field is empty
                if (!empty($field['required']) && empty($_POST['wvc_cpt_data'][$field_key])) {
                    // Skip saving if required field is empty - WordPress will show validation error
                    continue;
                }

                // Get value from POST
                $value = isset($_POST['wvc_cpt_data'][$field_key]) ? $_POST['wvc_cpt_data'][$field_key] : '';

                // Sanitize based on field type
                $sanitized_value = $this->sanitize_field_value($value, $field_type);

                // Save to individual meta key
                update_post_meta($post_id, $meta_key, $sanitized_value);
            }
        }
    }

    /**
     * Sanitize field value based on field type
     *
     * @param mixed  $value      The field value.
     * @param string $field_type The field type.
     *
     * @return mixed Sanitized value
     */
    private function sanitize_field_value($value, $field_type)
    {
        switch ($field_type) {
            case 'text':
            case 'select':
                return sanitize_text_field($value);

            case 'textarea':
                return sanitize_textarea_field($value);

            case 'url':
                return esc_url_raw($value);

            case 'email':
                return sanitize_email($value);

            case 'number':
            case 'image':
                return intval($value);

            case 'checkbox':
                return !empty($value) ? 1 : 0;

            case 'date':
                return sanitize_text_field($value); // Date format validation could be added

            default:
                return sanitize_text_field($value);
        }
    }

    /**
     * Enqueue scripts for image upload functionality
     *
     * @param string $hook The current admin page hook.
     */
    public function enqueue_scripts($hook)
    {
        // Only load on post edit screens
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }

        // Check if current post type has custom fields
        global $post;
        if (!$post) {
            return;
        }

        $fields = get_option("wvc_cpt_fields_{$post->post_type}", array());

        // Only enqueue if there are image fields
        $has_image_field = false;
        foreach ($fields as $field) {
            if ($field['type'] === 'image') {
                $has_image_field = true;
                break;
            }
        }

        if (!$has_image_field) {
            return;
        }

        // Enqueue WordPress media uploader
        wp_enqueue_media();

        // Enqueue custom script for image upload
        wp_add_inline_script('jquery', "
            jQuery(document).ready(function($) {
                // Upload image button
                $('.wvc-upload-image-button').on('click', function(e) {
                    e.preventDefault();

                    var button = $(this);
                    var fieldId = button.data('field-id');
                    var field = $('#' + fieldId);
                    var preview = button.siblings('.wvc-image-preview');
                    var removeButton = button.siblings('.wvc-remove-image-button');

                    var mediaUploader = wp.media({
                        title: 'Select Image',
                        button: {
                            text: 'Use this image'
                        },
                        multiple: false
                    });

                    mediaUploader.on('select', function() {
                        var attachment = mediaUploader.state().get('selection').first().toJSON();
                        field.val(attachment.id);
                        preview.html('<img src=\"' + attachment.url + '\" style=\"max-width: 300px; height: auto; display: block;\" />');
                        removeButton.show();
                    });

                    mediaUploader.open();
                });

                // Remove image button
                $('.wvc-remove-image-button').on('click', function(e) {
                    e.preventDefault();

                    var button = $(this);
                    var fieldId = button.data('field-id');
                    var field = $('#' + fieldId);
                    var preview = button.siblings('.wvc-image-preview');

                    field.val('');
                    preview.html('');
                    button.hide();
                });
            });
        ");
    }
}

// Initialize the CPT Meta Box Handler
WVC_CPT_Meta_Box::get_instance();